﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Globalization;
using System.Windows.Forms;
using System.IO;

using Ivi.Driver.Interop;
using Agilent.AgM938x.Interop;

namespace NSM9391A
{
    public partial class Vsag_M9391A
    {

        #region Load Waveforms

        public void loadVsgWaveform(string filePath, string fileName)
        {
            try
            {
                if (fileName.Contains(".wfm") || fileName.Contains(".SECUREWAVE"))
                {
                    // Use the driver method to load normal encrypted signal studio files
                    Vsg.Modulation.IQ.UploadArbAgilentFile(fileName, filePath + fileName);
                }
                else if (fileName.Contains(".WAVEFORM"))
                {
                    // These are unencrypted wavefroms from the MXG.  Each waveform will have three files:
                    // .WAVEFORM :  Contains the IQ data
                    // .HEADER : Contains header information, we need to get the sample rate and RMS value from this file
                    // .MARKER : Contains ALC and pulse blanking marker data

                    // Read Header, marker and waveform files, and format for load from array driver method
                    FileInfo fi = new FileInfo(filePath + fileName);
                    long waveformSize = fi.Length / 2;
                    // Read the waveform file (.WAVEFORM)
                    var ws = new FileStream(filePath + fileName, FileMode.Open);
                    BinaryReader wd = new BinaryReader(ws);
                    double[] waveformData = new double[waveformSize];
                    byte[] a16 = new byte[2];
                    for (int i = 0; i < waveformSize; i++)
                    {
                        // Byte swap Int16, then divide by 32768 to scale from -1 to 1
                        a16 = wd.ReadBytes(2);
                        Array.Reverse(a16);
                        waveformData[i] = (double)(BitConverter.ToInt16(a16, 0)) / 32768;
                    }
                    wd.Close();
                    ws.Close();
                    // Read the marker data from the .MARKERS file
                    string markerFileName = fileName.Replace(".WAVEFORM", ".MARKERS");
                    var ms = new FileStream(filePath + markerFileName, FileMode.Open);
                    BinaryReader md = new BinaryReader(ms);
                    byte[] markerData = new byte[waveformSize / 2];
                    for (int i = 0; i < waveformSize / 2; i++) markerData[i] = md.ReadByte();
                    md.Close();
                    ms.Close();
                    // Read the required data from the .HEADER file
                    string headerFileName = fileName.Replace(".WAVEFORM", ".HEADER");
                    var hs = new FileStream(filePath + headerFileName, FileMode.Open);
                    BinaryReader hd = new BinaryReader(hs);
                    // Header Structure:
                    // skip 50 bytes
                    // Read double for Sample rate
                    // skip 4 bytes
                    // read u8 for ALC Routing
                    // skip 1 byte
                    // read u8 for pulse routing
                    // skip 8 bytes
                    // read double for RMS value
                    // read float for scale factor
                    for (int i = 0; i < 50; i++) hd.ReadByte(); // throw these away
                    byte[] a64 = new byte[8];
                    a64 = hd.ReadBytes(8);
                    Array.Reverse(a64);
                    arbSampleRate = BitConverter.ToDouble(a64, 0);

                    for (int i = 0; i < 4; i++) hd.ReadByte(); // throw these away
                    byte markerNum = hd.ReadByte();
                    AgM938xMarkerEnum pulseMarker;
                    switch (markerNum)
                    {
                        case 1:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker1;
                            break;
                        case 2:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker2;
                            break;
                        case 3:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker3;
                            break;
                        case 4:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker4;
                            break;
                        default:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarkerNone;
                            break;
                    }

                    hd.ReadByte(); // skip 1
                    markerNum = hd.ReadByte();
                    AgM938xMarkerEnum alcMarker;
                    switch (markerNum)
                    {
                        case 1:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker1;
                            break;
                        case 2:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker2;
                            break;
                        case 3:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker3;
                            break;
                        case 4:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker4;
                            break;
                        default:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarkerNone;
                            break;
                    }

                    for (int i = 0; i < 8; i++) hd.ReadByte(); // throw these away
                    a64 = hd.ReadBytes(8);
                    Array.Reverse(a64);
                    arbRmsValue = -20 * Math.Log10(BitConverter.ToDouble(a64, 0));  // -20*Log10 of value in file
                    byte[] a32 = new byte[4];
                    a32 = hd.ReadBytes(4);
                    Array.Reverse(a32);
                    arbScale = BitConverter.ToSingle(a32, 0) / 100; // Percent in file, divide by 100
                    hd.Close();
                    hs.Close();

                    // Finally, load the waveform data to the M9381A
                    Vsg.Modulation.IQ.UploadArbDoublesWithMarkers(fileName, ref waveformData, ref markerData,
                        arbSampleRate, arbRmsValue, arbScale, pulseMarker, alcMarker);
                }

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        public void loadShortVsgWaveform(string filePath, string fileName, double startTime, double timeToLoad)
        {
            try
            {
                if (fileName.Contains(".wfm") || fileName.Contains(".SECUREWAVE"))
                {
                    Exception ex = new Exception("Can not load short version of Signal Studio Waveform");
                    throw (ex);
                }
                else if (fileName.Contains(".WAVEFORM"))
                {
                    // Read Header, marker and waveform files, and format for load from array driver method
                    // Read the required data from the .HEADER file
                    string headerFileName = fileName.Replace(".WAVEFORM", ".HEADER");
                    var hs = new FileStream(filePath + headerFileName, FileMode.Open);
                    BinaryReader hd = new BinaryReader(hs);
                    // Header Structure:
                    // skip 50 bytes
                    // Read double for Sample rate
                    // skip 4 bytes
                    // read u8 for ALC Routing
                    // skip 1 byte
                    // read u8 for pulse routing
                    // skip 8 bytes
                    // read double for RMS value
                    // read float for scale factor
                    for (int i = 0; i < 50; i++) hd.ReadByte(); // throw these away
                    byte[] a64 = new byte[8];
                    a64 = hd.ReadBytes(8);
                    Array.Reverse(a64);
                    arbSampleRate = BitConverter.ToDouble(a64, 0);

                    for (int i = 0; i < 4; i++) hd.ReadByte(); // throw these away
                    byte markerNum = hd.ReadByte();
                    AgM938xMarkerEnum pulseMarker;
                    switch (markerNum)
                    {
                        case 1:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker1;
                            break;
                        case 2:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker2;
                            break;
                        case 3:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker3;
                            break;
                        case 4:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarker4;
                            break;
                        default:
                            pulseMarker = AgM938xMarkerEnum.AgM938xMarkerNone;
                            break;
                    }

                    hd.ReadByte(); // skip 1
                    markerNum = hd.ReadByte();
                    AgM938xMarkerEnum alcMarker;
                    switch (markerNum)
                    {
                        case 1:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker1;
                            break;
                        case 2:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker2;
                            break;
                        case 3:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker3;
                            break;
                        case 4:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarker4;
                            break;
                        default:
                            alcMarker = AgM938xMarkerEnum.AgM938xMarkerNone;
                            break;
                    }

                    for (int i = 0; i < 8; i++) hd.ReadByte(); // throw these away
                    a64 = hd.ReadBytes(8);
                    Array.Reverse(a64);
                    arbRmsValue = -20 * Math.Log10(BitConverter.ToDouble(a64, 0));  // -20*Log10 of value in file
                    byte[] a32 = new byte[4];
                    a32 = hd.ReadBytes(4);
                    Array.Reverse(a32);
                    arbScale = BitConverter.ToSingle(a32, 0) / 100; // Percent in file, divide by 100
                    hd.Close();
                    hs.Close();

                    FileInfo fi = new FileInfo(filePath + fileName);
                    long waveformSize = fi.Length / 2;
                    double waveformTime = waveformSize / arbSampleRate;
                    if (waveformTime < timeToLoad + startTime)
                    {
                        Exception ex = new Exception("Waveform shorter than requested time");
                        throw (ex);
                    }
                    waveformSize = (int)(timeToLoad * arbSampleRate * 2);
                    int samplesToSkip = (int)(startTime * arbSampleRate * 2);

                    // Read the waveform file (.bin)
                    var ws = new FileStream(filePath + fileName, FileMode.Open);
                    BinaryReader wd = new BinaryReader(ws);
                    double[] waveformData = new double[waveformSize];
                    // Skip the beginning
                    for (int i = 0; i < samplesToSkip; i++) wd.ReadBytes(2); // throw these away
                    byte[] a16 = new byte[2];
                    for (int i = 0; i < waveformSize; i++)
                    {
                        // Byte swap Int16, then divide by 32768 to scale from -1 to 1
                        a16 = wd.ReadBytes(2);
                        Array.Reverse(a16);
                        waveformData[i] = (double)(BitConverter.ToInt16(a16, 0)) / 32768;
                    }
                    wd.Close();
                    ws.Close();

                    // Read the marker data from the .MARKERS file
                    string markerFileName = fileName.Replace(".WAVEFORM", ".MARKERS");
                    var ms = new FileStream(filePath + markerFileName, FileMode.Open);
                    BinaryReader md = new BinaryReader(ms);
                    byte[] markerData = new byte[waveformSize / 2];
                    for (int i = 0; i < samplesToSkip; i++) md.ReadByte(); // throw these away
                    for (int i = 0; i < waveformSize / 2; i++)
                    {
                        markerData[i] = md.ReadByte();
                    }
                    md.Close();
                    ms.Close();

                    // Finally, load the waveform data to the M9381A
                    Vsg.Modulation.IQ.UploadArbDoublesWithMarkers(fileName + "Short", ref waveformData, ref markerData,
                        arbSampleRate, arbRmsValue, arbScale, pulseMarker, alcMarker);
                }

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        #endregion

        #region Create Waveforms

        // generatePowerRampArb
        // Generate a CW ARB file (using DC values on I and Q)
        // Vary the amplitude based on min and max power fields
        public void generatePowerRampArb(string refName, double minPower, double maxPower, double duration, double sampleRate)
        {
            // Determine number of points
            int numPoints = (int)(duration * sampleRate);

            // Calcualte the scale factors
            double amplitude = Math.Sqrt(2) / 2;
            double scaleIncr = (maxPower - minPower) / (numPoints);

            // Create waveform and build array
            double[] waveform = new double[numPoints * 4];
            for (int i = 0; i < waveform.Length / 4; i++)
            {
                // Create I and Q and correct amplitude
                waveform[2 * i] = amplitude * Math.Pow(10, (minPower + i * scaleIncr) / 20);
                waveform[2 * i + 1] = waveform[2 * i];
            }
            for (int i = 0; i < waveform.Length / 4; i++)
            {
                // Create I and Q and correct amplitude
                waveform[waveform.Length / 2 + 2 * i] = amplitude * Math.Pow(10, (maxPower - i * scaleIncr) / 20);
                waveform[waveform.Length / 2 + 2 * i + 1] = waveform[waveform.Length / 2 + 2 * i];
            }
            Vsg.Modulation.IQ.UploadArbDoubles(refName, ref waveform, sampleRate, 0, 0.9);
        }

        // generatePulseArbSequence
        // Generate an ARB sequence named "PulseSequence" around a loaded ARB file 
        // by adding zeros to beginning and end of waveform
        public void generatePulseArbSequence(string refName, double leadTime, double dutyCycle)
        {
            // Get Sample Rate of ARB File
            Vsg.Modulation.IQ.ArbInformation(refName, ref arbSampleRate, ref arbRmsValue, arbScale, rfMarker, alcMarker);

            // Delete any exisiting pulse sequence
            Vsg.Modulation.Sequence.Remove("Sequence");

            // Create the Leading Zero waveform
            int numPoints = (int)(leadTime * arbSampleRate);
            double[] waveform = new double[numPoints * 2];
            Array.Clear(waveform, 0, waveform.Length);
            Vsg.Modulation.IQ.RemoveArb("LeadingZeros");
            if (leadTime > 0)
                Vsg.Modulation.IQ.UploadArbDoubles("LeadingZeros", ref waveform, arbSampleRate, 0, 1);

            // Create the Trailing Zero Waveform 
            // Trailing time = (PulseWidth/DC) - PulseWidth - LeadTime
            // WLAN Waveforms are 100% duty Cycle, Get Pulse Width from number of samples
            double rfPulsewidth = getNumberSamples(refName) / arbSampleRate;
            double trailTime = (rfPulsewidth / dutyCycle) - rfPulsewidth - leadTime;
            waveform = null;
            numPoints = (int)(trailTime * arbSampleRate);
            waveform = new double[numPoints * 2];
            Array.Clear(waveform, 0, waveform.Length);
            Vsg.Modulation.IQ.RemoveArb("TrailingZeros");
            Vsg.Modulation.IQ.UploadArbDoubles("TrailingZeros", ref waveform, arbSampleRate, 0, 1);

            // Create the Sequence
            Vsg.Modulation.Sequence.Create("Sequence");
            if (leadTime > 0)
            {
                Vsg.Modulation.Sequence.AddStep("LeadingZeros");
                Vsg.Modulation.Sequence.AddSegment("LeadingZeros");
                Vsg.Modulation.Sequence.StepRepetitions = 1;
            }
            Vsg.Modulation.Sequence.AddStep(refName);
            Vsg.Modulation.Sequence.AddSegment(refName);
            Vsg.Modulation.Sequence.StepRepetitions = 1;
            Vsg.Modulation.Sequence.AddStep("TrailingZeros");
            Vsg.Modulation.Sequence.AddSegment("TrailingZeros");
            Vsg.Modulation.Sequence.StepRepetitions = 1;
            Vsg.Modulation.Sequence.End();
        }

        public int getNumberSamples(string refName)
        {
            int numSamples = 0;
            // Use ARB Catalog to get numbe rof samples
            string arbCatalog = Vsg.Modulation.CatalogContents();
            // Parse:  find Refernce name, then slit on comma, second value is number of samples
            arbCatalog = arbCatalog.Remove(0, arbCatalog.IndexOf(refName, 0));
            string[] catalogArray = arbCatalog.Split(',');
            numSamples = Convert.ToInt16(catalogArray[1]);
            return numSamples;
        }
        #endregion
    }
}